package org.stianloader.picoresolve.internal.meta;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import org.jetbrains.annotations.NotNull;

/**
 * The *.lastUpdated file are files generated by the maven resolver to describe when a file was fetched from the
 * remote repository the last time.
 */
public class LastUpdatedFile {

    private final Map<String, String> errors = new HashMap<>();
    private final Map<String, Long> lastFetch = new HashMap<>();
    private final Map<Object, Object> nonsensePairs = new HashMap<>();

    public LastUpdatedFile updateEntryErrored(String repo, String error, long updateTime) {
        this.errors.put(repo, error);
        this.lastFetch.put(repo, updateTime);
        return this;
    }

    public LastUpdatedFile updateEntrySuccess(String repo, long updateTime) {
        this.errors.remove(repo);
        this.lastFetch.put(repo, updateTime);
        return this;
    }

    public boolean hasErrored(String repoURL) {
        return this.errors.containsKey(repoURL);
    }

    public Long getLastFetchTime(String repoURL) {
        return this.lastFetch.get(repoURL);
    }

    public static LastUpdatedFile tryParse(@NotNull Path src) {
        LastUpdatedFile f = new LastUpdatedFile();
        if (Files.notExists(src)) {
            return f;
        }
        Properties prop = new Properties();
        try (InputStream is = Files.newInputStream(src)) {
            prop.load(is);
        } catch (IOException ignored) {
            ignored.printStackTrace();
            return f;
        }
        prop.forEach((key, value) -> {
            String keyString = key.toString();
            String valueString = value.toString();
            int dotIndex = keyString.indexOf('.');
            if (dotIndex != -1) {
                String repo = keyString.substring(0, dotIndex);
                String action = keyString.substring(dotIndex + 1);
                if (action.equals("error")) {
                    f.errors.put(repo, valueString);
                } else if (action.equals("lastUpdated")) {
                    f.lastFetch.put(repo, Long.valueOf(valueString));
                } else {
                    f.nonsensePairs.put(key, value);
                }
            } else {
                f.nonsensePairs.put(key, value);
            }
        });
        return f;
    }

    public void write(@NotNull Path out) {
        Properties props = new Properties();
        this.errors.forEach((key, val) -> {
            props.put(key + ".error", val);
        });
        this.lastFetch.forEach((key, val) -> {
            props.put(key + ".lastUpdated", val.toString());
        });
        this.nonsensePairs.forEach(props::put);
        try (OutputStream os = Files.newOutputStream(out)) {
            props.store(os, "NOTE: This is a Maven Resolver internal implementation file, its format can be changed without prior notice.\n"
                    + "NOTE: This file was written by picoresolve, a nonstandard resolver implementation!");
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
}
